# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


import os
from sendgrid import SendGridAPIClient
from sendgrid.helpers.mail import Mail, Email, To, Content

class SendGridAPI:
    def __init__(self, api_key: str):
        self.sg = SendGridAPIClient(api_key)
        self.from_email = os.getenv("SENDGRID_FROM_EMAIL", "noreply@example.com") # Default fallback

    def send_email(self, to_emails: list[str], subject: str, html_content: str):
        """
        Sends an email using SendGrid.

        Args:
            to_emails: A list of recipient email addresses.
            subject: The subject of the email.
            html_content: The HTML content of the email.
        """
        if not to_emails:
            print("No recipients provided. Email not sent.")
            return

        message = Mail(
            from_email=Email(self.from_email),
            to_emails=[To(email) for email in to_emails],
            subject=subject,
            html_content=Content("text/html", html_content)
        )

        try:
            response = self.sg.send(message)
            print(f"Email sent successfully to {', '.join(to_emails)}.")
            print(f"SendGrid Response Status Code: {response.status_code}")
            # print(f"SendGrid Response Body: {response.body}") # Uncomment for detailed response
            # print(f"SendGrid Response Headers: {response.headers}") # Uncomment for detailed response
        except Exception as e:
            print(f"Error sending email via SendGrid: {e}")

    def send_transactional_email(self, to_email: str, template_id: str, dynamic_template_data: dict):
        """
        Sends a transactional email using a SendGrid template.

        Args:
            to_email: The recipient email address.
            template_id: The ID of the SendGrid dynamic template.
            dynamic_template_data: A dictionary containing data to populate the template.
        """
        message = Mail(
            from_email=Email(self.from_email),
            to_emails=[To(to_email)],
            subject="", # Subject is typically defined in the template
            html_content=""
        )
        message.template_id = template_id
        message.dynamic_template_data = dynamic_template_data

        try:
            response = self.sg.send(message)
            print(f"Transactional email sent successfully to {to_email} using template {template_id}.")
            print(f"SendGrid Response Status Code: {response.status_code}")
        except Exception as e:
            print(f"Error sending transactional email via SendGrid: {e}")

