# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


from google.analytics.data_v1beta import BetaAnalyticsDataClient
from google.analytics.data_v1beta.types import (
    DateRange,
    Dimension,
    Metric,
    RunReportRequest,
)
import os

class GoogleAnalyticsAPI:
    def __init__(self, property_id: str, view_id: str):
        # Note: view_id is for Universal Analytics (UA). For GA4, property_id is sufficient.
        # This implementation assumes GA4 properties and uses property_id.
        # If using UA, you'd need to adjust the client and request accordingly.
        self.property_id = property_id
        # self.view_id = view_id # Not directly used in GA4 API calls
        self.client = BetaAnalyticsDataClient()

    def get_campaign_metrics(self, campaign_id: str, days: int = 30) -> dict:
        """
        Retrieves key engagement metrics for a specific campaign from Google Analytics.

        Args:
            campaign_id: The identifier for the marketing campaign.
            days: The number of past days to retrieve data for.

        Returns:
            A dictionary containing campaign metrics (e.g., sessions, users, conversions).
        """
        # Construct the campaign name filter. This assumes your campaign names in GA
        # start with the campaign_id or contain it in a predictable way.
        # You might need to adjust this filter based on your GA naming conventions.
        campaign_filter = f"sessionCampaignName::{campaign_id}"

        request = RunReportRequest(
            property=f"properties/{self.property_id}",
            dimensions=[
                Dimension(name="sessionCampaignName"),
                Dimension(name="date")
            ],
            metrics=[
                Metric(name="sessions"),
                Metric(name="totalUsers"),
                Metric(name="conversions"), # Use appropriate metric for your goals
                Metric(name="eventCount") # Example for general event tracking
            ],
            date_ranges=[
                DateRange(start_date=f"-{days}days", end_date="today")
            ],
            dimension_filter= {
                "field_name": "sessionCampaignName",
                "string_filter": {
                    "match_type": "EXACT",
                    "value": campaign_id
                }
            }
        )

        try:
            response = self.client.run_report(request=request)
            
            metrics_data = {
                "campaign_id": campaign_id,
                "total_sessions": 0,
                "total_users": 0,
                "total_conversions": 0,
                "total_events": 0,
                "daily_breakdown": []
            }

            for row in response.rows:
                date = row.dimension_values[1].value # Date is the second dimension
                sessions = int(row.metric_values[0].value)
                users = int(row.metric_values[1].value)
                conversions = int(row.metric_values[2].value)
                events = int(row.metric_values[3].value)

                metrics_data["total_sessions"] += sessions
                metrics_data["total_users"] += users
                metrics_data["total_conversions"] += conversions
                metrics_data["total_events"] += events

                metrics_data["daily_breakdown"].append({
                    "date": date,
                    "sessions": sessions,
                    "users": users,
                    "conversions": conversions,
                    "events": events
                })
            
            # Calculate averages if needed
            if metrics_data["total_sessions"] > 0:
                metrics_data["average_conversions_per_session"] = round(metrics_data["total_conversions"] / metrics_data["total_sessions"], 2)
            else:
                metrics_data["average_conversions_per_session"] = 0

            return metrics_data

        except Exception as e:
            print(f"Error retrieving Google Analytics data for campaign '{campaign_id}': {e}")
            return {"error": str(e)}

    def get_user_engagement_summary(self, user_id: str) -> dict:
        """
        Retrieves a summary of engagement for a specific user.
        This is a placeholder and would require custom dimensions or user-specific reporting.
        """
        print(f"Fetching engagement summary for user: {user_id} (Placeholder)")
        # In a real GA4 setup, you might use User ID reporting if configured.
        # For this example, we return dummy data.
        return {
            "user_id": user_id,
            "total_sessions": 15,
            "total_pages_viewed": 75,
            "last_session_date": "2023-10-26"
        }
