# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


import unittest
from unittest.mock import MagicMock

from langchain_google_genai import ChatGoogleGenerativeAI
from mailing_list_engagement.agents.reporting_agent import ReportingAgent
from mailing_list_engagement.tools.google_analytics_api import GoogleAnalyticsAPI

class TestReportingAgent(unittest.TestCase):

    def setUp(self):
        """Set up test environment."""
        self.mock_llm = MagicMock(spec=ChatGoogleGenerativeAI)
        self.mock_analytics_tool = MagicMock(spec=GoogleAnalyticsAPI)
        self.reporting_agent = ReportingAgent(llm=self.mock_llm, analytics_tool=self.mock_analytics_tool)

    def test_generate_report(self):
        """Test if the reporting agent generates a report correctly."""
        campaign_goal = "Increase open rates by 10%."
        engagement_metrics = {
            "campaign_id": "q3_promo",
            "total_sessions": 2000,
            "total_users": 1800,
            "total_conversions": 250,
            "total_events": 6000,
            "average_conversions_per_session": 0.125
        }
        expected_report = "Campaign Report:\nGoal: Increase open rates by 10%.\nPerformance: Achieved 12.5% conversion rate, exceeding the target. Total sessions: 2000, Total users: 1800."

        self.mock_llm.invoke.return_value.content = expected_report

        report = self.reporting_agent.generate_report(campaign_goal, engagement_metrics)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(report, expected_report)

    def test_generate_report_with_recommendations(self):
        """Test report generation when recommendations might be included."""
        campaign_goal = "Boost click-through rates."
        engagement_metrics = {
            "campaign_id": "cta_boost",
            "total_sessions": 1000,
            "total_users": 900,
            "total_conversions": 50,
            "total_events": 3000,
            "average_conversions_per_session": 0.05
        }
        expected_report = "Campaign Report:\nGoal: Boost click-through rates.\nPerformance: Click-through rate was 5%. Consider A/B testing different call-to-action buttons."

        self.mock_llm.invoke.return_value.content = expected_report

        report = self.reporting_agent.generate_report(campaign_goal, engagement_metrics)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(report, expected_report)

    def test_generate_report_no_metrics(self):
        """Test report generation with empty metrics."""
        campaign_goal = "Test campaign with no metrics."
        engagement_metrics = {}
        expected_report = "Campaign Report:\nGoal: Test campaign with no metrics.\nPerformance: No engagement metrics available to report."

        self.mock_llm.invoke.return_value.content = expected_report

        report = self.reporting_agent.generate_report(campaign_goal, engagement_metrics)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(report, expected_report)

if __name__ == '__main__':
    unittest.main()
