# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


import os
from dotenv import load_dotenv
from langchain_google_genai import ChatGoogleGenerativeAI
from langchain_core.prompts import ChatPromptTemplate
from langchain_core.runnables import RunnablePassthrough

from mailing_list_engagement.agents.content_creation_agent import ContentCreationAgent
from mailing_list_engagement.agents.engagement_analysis_agent import EngagementAnalysisAgent
from mailing_list_engagement.agents.list_management_agent import ListManagementAgent
from mailing_list_engagement.agents.personalization_agent import PersonalizationAgent
from mailing_list_engagement.agents.reporting_agent import ReportingAgent
from mailing_list_engagement.tools.sendgrid_api import SendGridAPI
from mailing_list_engagement.tools.vertex_ai_tools import VertexAITools
from mailing_list_engagement.tools.google_analytics_api import GoogleAnalyticsAPI

# Load environment variables
load_dotenv()

# Initialize LLM
llm = ChatGoogleGenerativeAI(
    model="gemini-2.5-flash",
    google_api_key=os.getenv("GOOGLE_API_KEY")
)

# Initialize Tools
sendgrid_tool = SendGridAPI(api_key=os.getenv("SENDGRID_API_KEY"))
vertex_ai_tool = VertexAITools(project_id=os.getenv("VERTEX_AI_PROJECT_ID"), location=os.getenv("VERTEX_AI_LOCATION"))
google_analytics_tool = GoogleAnalyticsAPI(
    property_id=os.getenv("GOOGLE_ANALYTICS_PROPERTY_ID"),
    view_id=os.getenv("GOOGLE_ANALYTICS_VIEW_ID")
)

# Initialize Agents
content_creation_agent = ContentCreationAgent(llm=llm)
engagement_analysis_agent = EngagementAnalysisAgent(llm=llm, analytics_tool=google_analytics_tool)
list_management_agent = ListManagementAgent(llm=llm)
personalization_agent = PersonalizationAgent(llm=llm)
reporting_agent = ReportingAgent(llm=llm, analytics_tool=google_analytics_tool)

def run_campaign_workflow():
    print("Starting mailing list engagement campaign...")

    # 1. Define campaign goals and target audience (simplified for example)
    campaign_goal = "Increase open rates for the latest product announcement."
    target_audience = "Existing customers who have purchased in the last 6 months."
    print(f"\nCampaign Goal: {campaign_goal}")
    print(f"Target Audience: {target_audience}")

    # 2. Generate email content
    print("\nGenerating email content...")
    email_content = content_creation_agent.create_content(campaign_goal, target_audience)
    print(f"Generated Content:\n{email_content}")

    # 3. Personalize email content
    print("\nPersonalizing email content...")
    personalized_content = personalization_agent.personalize_content(email_content, target_audience)
    print(f"Personalized Content:\n{personalized_content}")

    # 4. Send emails (using SendGrid - placeholder for actual sending)
    print("\nSending emails via SendGrid...")
    # In a real scenario, you would fetch a list of recipients from list_management_agent
    # and then use sendgrid_tool.send_email(recipients, personalized_content)
    print("Emails sent (simulated).")

    # 5. Analyze engagement (after some time has passed for opens/clicks)
    print("\nAnalyzing engagement metrics...")
    engagement_metrics = engagement_analysis_agent.analyze_engagement(campaign_id="product_announcement_q3")
    print(f"Engagement Metrics:\n{engagement_metrics}")

    # 6. Generate report
    print("\nGenerating campaign report...")
    report = reporting_agent.generate_report(campaign_goal, engagement_metrics)
    print(f"Campaign Report:\n{report}")

    print("\nMailing list engagement campaign finished.")

if __name__ == "__main__":
    run_campaign_workflow()
