# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


import os
import unittest
import subprocess

# Assume the main application script is 'main.py' and it can be run
# We will try to execute it and check for basic output or errors.

class TestDeployment(unittest.TestCase):

    def setUp(self):
        """Set up test environment."""
        # Ensure required environment variables are set for the test
        self.required_env_vars = [
            "GOOGLE_API_KEY",
            "SENDGRID_API_KEY",
            "GOOGLE_ANALYTICS_PROPERTY_ID",
            "VERTEX_AI_PROJECT_ID",
            "VERTEX_AI_LOCATION"
        ]
        self.missing_vars = [var for var in self.required_env_vars if not os.getenv(var)]
        if self.missing_vars:
            self.skipTest(f"Skipping deployment tests due to missing environment variables: {', '.join(self.missing_vars)}")

        self.main_script = os.path.join(os.path.dirname(__file__), '..', 'main.py')

    def test_main_script_execution(self):
        """Test if the main script can be executed without immediate errors."""
        if self.missing_vars: return # Skip if setup failed

        try:
            # Execute the main script and capture output/errors
            # We expect some output indicating the workflow is starting/running.
            # A timeout is useful to prevent hanging.
            result = subprocess.run(
                ['python', self.main_script],
                capture_output=True,
                text=True,
                check=True, # Raises CalledProcessError if return code is non-zero
                timeout=60 # 60 seconds timeout
            )
            print(f"\n--- STDOUT ---\n{result.stdout}\n--- STDERR ---\n{result.stderr}")
            self.assertIn("Mailing list engagement campaign finished.", result.stdout, "Expected completion message not found in stdout.")
            self.assertIn("Starting mailing list engagement campaign...", result.stdout, "Expected start message not found in stdout.")

        except FileNotFoundError:
            self.fail(f"Python interpreter or script '{self.main_script}' not found.")
        except subprocess.CalledProcessError as e:
            self.fail(f"Script execution failed with return code {e.returncode}:\nSTDOUT:\n{e.stdout}\nSTDERR:\n{e.stderr}")
        except subprocess.TimeoutExpired:
            self.fail("Script execution timed out.")
        except Exception as e:
            self.fail(f"An unexpected error occurred during script execution: {e}")

    # Add more tests here for specific deployment scenarios, e.g.,
    # - Testing API integrations if they can be mocked or called in a test environment.
    # - Testing cloud resource creation/deletion (requires more complex setup).

if __name__ == '__main__':
    unittest.main()
