# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


# tools/lending_protocol_api.py
import os
from web3 import Web3
from dotenv import load_dotenv

load_dotenv()

# --- Configuration ---
# Use a reliable RPC provider for Base Mainnet
# Example: Use a free tier from Alchemy or Infura, or a paid service for production.
RPC_URL = os.getenv("BASE_RPC_URL")
if not RPC_URL:
    raise ValueError("BASE_RPC_URL environment variable not set.")

# Connect to Base chain
try:
    w3 = Web3(Web3.HTTPProvider(RPC_URL))
    if not w3.is_connected():
        raise Exception("Not connected to Base blockchain!")
except Exception as e:
    raise Exception(f"Failed to connect to Base blockchain: {e}")

# --- Contract Details ---
# Replace with actual Base Mainnet USDC contract address
# Source: https://docs.base.org/tokens
USDC_CONTRACT_ADDRESS = "0x833589fcd6edb6e08f4c7c32d4f71b54b9799745" # USDC on Base

# Replace with the actual contract address of the lending protocol you are using on Base.
# Example: Aave V3 on Base (address may change, verify on BaseScan)
LENDING_PROTOCOL_CONTRACT_ADDRESS = "0x7943513701706336859307666893763666475576" # Example: Aave V3 Pool on Base

# Replace with the actual ABI for the USDC contract on Base.
# You can find this on BaseScan.
USDC_ABI = [
    {
        "constant": True,
        "inputs": [{"name": "_owner", "type": "address"}],
        "name": "balanceOf",
        "outputs": [{"name": "balance", "type": "uint256"}],
        "type": "function"
    },
    {
        "constant": False,
        "inputs": [
            {"name": "_spender", "type": "address"},
            {"name": "_value", "type": "uint256"}
        ],
        "name": "approve",
        "outputs": [{"name": "success", "type": "bool"}],
        "type": "function"
    }
]

# Replace with the actual ABI for the lending protocol's contract.
# This is a simplified example for Aave V3. You'll need the full ABI.
# Source: Aave V3 Pool ABI (verify on BaseScan or Aave docs)
LENDING_PROTOCOL_ABI = [
    {
        "inputs": [
            {"internalType": "address", "name": "asset", "type": "address"},
            {"internalType": "uint256", "name": "amount", "type": "uint256"}
        ],
        "name": "deposit",
        "outputs": [],
        "stateMutability": "nonpayable",
        "type": "function"
    },
    {
        "inputs": [
            {"internalType": "address", "name": "asset", "type": "address"},
            {"internalType": "uint256", "name": "amount", "type": "uint256"},
            {"internalType": "uint16", "name": "referralCode", "type": "uint16"}
        ],
        "name": "withdraw",
        "outputs": [],
        "stateMutability": "nonpayable",
        "type": "function"
    },
    {
        "inputs": [
            {"internalType": "address", "name": "user", "type": "address"},
            {"internalType": "address", "name": "asset", "type": "address"}
        ],
        "name": "balanceOf",
        "outputs": [{"internalType": "uint256", "name": "", "type": "uint256"}],
        "stateMutability": "view",
        "type": "function"
    }
]

# Initialize contracts
try:
    usdc_contract = w3.eth.contract(address=USDC_CONTRACT_ADDRESS, abi=USDC_ABI)
    lending_protocol_contract = w3.eth.contract(address=LENDING_PROTOCOL_CONTRACT_ADDRESS, abi=LENDING_PROTOCOL_ABI)
except Exception as e:
    raise Exception(f"Failed to initialize contracts: {e}")

# --- Helper Functions ---
def get_usdc_decimals():
    """Returns the number of decimals for USDC."""
    try:
        # USDC typically has 6 decimals on Base, but it's good practice to check.
        # This requires a contract call that might not be in the minimal ABI above.
        # For simplicity, we'll hardcode it or assume it's known.
        return 6
    except Exception as e:
        print(f"Warning: Could not fetch USDC decimals, assuming 6. Error: {e}")
        return 6

USDC_DECIMALS = get_usdc_decimals()

# --- Tool Implementations ---
def enter_lending_position(amount: float, lending_protocol: str, account_address: str, private_key: str):
    """Enters a USDC lending position on the Base blockchain."""
    try:
        if not w3.is_address(account_address):
            return "Invalid account address."
        if amount <= 0:
            return "Amount must be greater than zero."

        amount_wei = int(amount * (10**USDC_DECIMALS))

        # 1. Approve the lending protocol to spend USDC
        nonce = w3.eth.get_transaction_count(account_address)
        approve_tx = usdc_contract.functions.approve(LENDING_PROTOCOL_CONTRACT_ADDRESS, amount_wei).build_transaction({
            'chainId': w3.eth.chain_id,
            'gas': 200000, # Adjust gas limit as needed
            'gasPrice': w3.eth.gas_price,
            'nonce': nonce,
            'from': account_address
        })

        signed_approve_tx = w3.eth.account.sign_transaction(approve_tx, private_key)
        tx_hash_approve = w3.eth.send_raw_transaction(signed_approve_tx.rawTransaction)
        tx_receipt_approve = w3.eth.wait_for_transaction_receipt(tx_hash_approve)

        if tx_receipt_approve.get('status') == 0:
            return f"USDC Approval failed: {tx_receipt_approve['transactionHash'].hex()}"

        # 2. Deposit USDC into the lending protocol
        nonce += 1
        # The deposit function signature might vary. This assumes a simple deposit function.
        # For Aave V3, you might need to use the LendingPool's deposit function.
        deposit_tx = lending_protocol_contract.functions.deposit(USDC_CONTRACT_ADDRESS, amount_wei).build_transaction({
            'chainId': w3.eth.chain_id,
            'gas': 200000, # Adjust gas limit as needed
            'gasPrice': w3.eth.gas_price,
            'nonce': nonce,
            'from': account_address
        })

        signed_deposit_tx = w3.eth.account.sign_transaction(deposit_tx, private_key)
        tx_hash_deposit = w3.eth.send_raw_transaction(signed_deposit_tx.rawTransaction)
        tx_receipt_deposit = w3.eth.wait_for_transaction_receipt(tx_hash_deposit)

        if tx_receipt_deposit.get('status') == 0:
            return f"USDC Deposit failed: {tx_receipt_deposit['transactionHash'].hex()}"

        return f"Successfully entered lending position. Approval Tx: {tx_receipt_approve['transactionHash'].hex()}, Deposit Tx: {tx_receipt_deposit['transactionHash'].hex()}"

    except Exception as e:
        return f"Error entering lending position: {e}"

def adjust_lending_position(amount: float, lending_protocol: str, account_address: str, private_key: str, action: str = "withdraw"):
    """Adjusts a USDC lending position on the Base blockchain (deposit or withdraw)."""
    try:
        if not w3.is_address(account_address):
            return "Invalid account address."
        if amount <= 0:
            return "Amount must be greater than zero."

        amount_wei = int(amount * (10**USDC_DECIMALS))
        nonce = w3.eth.get_transaction_count(account_address)

        if action.lower() == "withdraw":
            # Example: Withdraw USDC from the lending protocol
            tx_func = lending_protocol_contract.functions.withdraw
            tx_params = [USDC_CONTRACT_ADDRESS, amount_wei]
        elif action.lower() == "deposit":
            # First, approve if depositing more
            approve_tx = usdc_contract.functions.approve(LENDING_PROTOCOL_CONTRACT_ADDRESS, amount_wei).build_transaction({
                'chainId': w3.eth.chain_id,
                'gas': 200000,
                'gasPrice': w3.eth.gas_price,
                'nonce': nonce,
                'from': account_address
            })
            signed_approve_tx = w3.eth.account.sign_transaction(approve_tx, private_key)
            tx_hash_approve = w3.eth.send_raw_transaction(signed_approve_tx.rawTransaction)
            tx_receipt_approve = w3.eth.wait_for_transaction_receipt(tx_hash_approve)
            if tx_receipt_approve.get('status') == 0:
                return f"USDC Approval failed for deposit adjustment: {tx_receipt_approve['transactionHash'].hex()}"
            nonce += 1
            tx_func = lending_protocol_contract.functions.deposit
            tx_params = [USDC_CONTRACT_ADDRESS, amount_wei]
        else:
            return "Invalid action. Use 'deposit' or 'withdraw'."

        adjust_tx = tx_func(*tx_params).build_transaction({
            'chainId': w3.eth.chain_id,
            'gas': 200000, # Adjust gas limit as needed
            'gasPrice': w3.eth.gas_price,
            'nonce': nonce,
            'from': account_address
        })

        signed_adjust_tx = w3.eth.account.sign_transaction(adjust_tx, private_key)
        tx_hash_adjust = w3.eth.send_raw_transaction(signed_adjust_tx.rawTransaction)
        tx_receipt_adjust = w3.eth.wait_for_transaction_receipt(tx_hash_adjust)

        if tx_receipt_adjust.get('status') == 0:
            return f"USDC {action.capitalize()} failed: {tx_receipt_adjust['transactionHash'].hex()}"

        return f"Successfully adjusted lending position ({action}). Tx Hash: {tx_receipt_adjust['transactionHash'].hex()}"

    except Exception as e:
        return f"Error adjusting lending position: {e}"

def get_current_position(account_address: str):
    """Retrieves the current details of the lending position."""
    try:
        if not w3.is_address(account_address):
            return "Invalid account address."

        # Get balance deposited in the lending protocol
        balance_wei = lending_protocol_contract.functions.balanceOf(account_address, USDC_CONTRACT_ADDRESS).call()
        balance = balance_wei / (10**USDC_DECIMALS)

        # Note: To get APY and other metrics, you would need to query the specific lending protocol's
        # contract or use a DeFi data aggregator API.
        return f"Current USDC balance in lending protocol: {balance:.6f} USDC"

    except Exception as e:
        return f"Error getting current position: {e}"
