# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


# tests/test_tools.py
import unittest
from unittest.mock import patch, MagicMock
import os

# Import the actual tool functions
from tools import blockchain_api, lending_protocol_api, risk_assessment_api

# Mock environment variables for tools that require them
@unittest.skipIf(not os.getenv("BASE_RPC_URL") or not os.getenv("ACCOUNT_ADDRESS") or not os.getenv("PRIVATE_KEY"), "Skipping tool tests: BASE_RPC_URL, ACCOUNT_ADDRESS, or PRIVATE_KEY not set.")
class TestTools(unittest.TestCase):

    def setUp(self):
        """Set up test environment."""
        # Load environment variables if not already loaded
        from dotenv import load_dotenv
        load_dotenv()

        # Mocking web3 connection for blockchain_api tests if needed
        self.mock_w3_connected = MagicMock()
        self.mock_w3_connected.is_connected.return_value = True
        self.mock_w3_connected.eth.gas_price = 1000000000 # Example gas price
        self.mock_w3_connected.eth.chain_id = 8453 # Base Mainnet Chain ID
        self.mock_w3_connected.to_wei = lambda value, unit: int(value * (10**6)) # Mock to_wei for USDC decimals
        self.mock_w3_connected.from_wei = lambda value, unit: value / (10**6) # Mock from_wei for USDC decimals
        self.mock_w3_connected.is_address = lambda addr: addr.startswith("0x") and len(addr) == 42

        # Patch web3.Web3 to return our mock instance for these tests
        self.patcher_web3 = patch('web3.Web3', return_value=self.mock_w3_connected)
        self.mock_web3 = self.patcher_web3.start()

        # Mock contract instances and their functions
        mock_usdc_contract = MagicMock()
        mock_lending_protocol_contract = MagicMock()
        
        # Mocking specific contract functions
        mock_usdc_contract.functions.approve.return_value.build_transaction.return_value = {'gas': 200000, 'gasPrice': self.mock_w3_connected.eth.gas_price, 'nonce': 0, 'from': os.getenv('ACCOUNT_ADDRESS'), 'chainId': self.mock_w3_connected.eth.chain_id}
        mock_lending_protocol_contract.functions.deposit.return_value.build_transaction.return_value = {'gas': 200000, 'gasPrice': self.mock_w3_connected.eth.gas_price, 'nonce': 1, 'from': os.getenv('ACCOUNT_ADDRESS'), 'chainId': self.mock_w3_connected.eth.chain_id}
        mock_lending_protocol_contract.functions.withdraw.return_value.build_transaction.return_value = {'gas': 200000, 'gasPrice': self.mock_w3_connected.eth.gas_price, 'nonce': 0, 'from': os.getenv('ACCOUNT_ADDRESS'), 'chainId': self.mock_w3_connected.eth.chain_id}
        mock_lending_protocol_contract.functions.balanceOf.return_value = 100000000 # Example balance (6 decimals)

        self.mock_web3.eth.contract.side_effect = [
            mock_usdc_contract, # First call returns usdc_contract
            mock_lending_protocol_contract # Second call returns lending_protocol_contract
        ]
        
        # Re-initialize the lending_protocol_api module to use the mocked web3 and contracts
        import importlib
        importlib.reload(lending_protocol_api)

    def tearDown(self):
        """Tear down test environment."""
        self.patcher_web3.stop()

    @patch('requests.get')
    def test_get_market_conditions(self, mock_requests_get):
        """Tests the get_market_conditions tool."""
        # Mock the requests.get call to return a successful response
        mock_response = MagicMock()
        mock_response.raise_for_status.return_value = None
        mock_response.json.return_value = {"status": "1", "message": "OK", "result": {"total_supply": "1000000000", "total_borrow": "500000000"}}
        mock_requests_get.return_value = mock_response
        
        result = blockchain_api.get_market_conditions()
        self.assertIsNotNone(result)
        self.assertIn("result", result)
        self.assertEqual(result['status'], "1")

    @patch('requests.get')
    def test_get_current_block(self, mock_requests_get):
        """Tests the get_current_block tool."""
        mock_response = MagicMock()
        mock_response.raise_for_status.return_value = None
        mock_response.json.return_value = {'jsonrpc': '2.0', 'id': 1, 'result': '0x12345'}
        mock_requests_get.return_value = mock_response

        result = blockchain_api.get_current_block()
        self.assertIsNotNone(result)
        self.assertIsInstance(result, int)
        self.assertEqual(result, 4660)

    def test_enter_lending_position_success(self):
        """Tests the enter_lending_position tool for success."""
        # Mocking transaction receipt status to be successful (1)
        self.mock_web3.eth.wait_for_transaction_receipt.side_effect = [
            {'status': 1, 'transactionHash': b'\x01' * 32},
            {'status': 1, 'transactionHash': b'\x02' * 32}
        ]
        
        amount = 100.5
        lending_protocol = "AaveV3"
        account_address = os.getenv("ACCOUNT_ADDRESS")
        private_key = os.getenv("PRIVATE_KEY")

        result = lending_protocol_api.enter_lending_position(amount, lending_protocol, account_address, private_key)
        
        self.assertIn("Successfully entered lending position.", result)
        self.assertIn("Approval Tx:", result)
        self.assertIn("Deposit Tx:", result)
        # Check if the correct functions were called
        self.mock_web3.eth.contract().functions.approve.assert_called_once()
        self.mock_web3.eth.contract().functions.deposit.assert_called_once()

    def test_enter_lending_position_approval_fail(self):
        """Tests enter_lending_position when approval fails."""
        self.mock_web3.eth.wait_for_transaction_receipt.side_effect = [
            {'status': 0, 'transactionHash': b'\x01' * 32},
            {'status': 1, 'transactionHash': b'\x02' * 32} # This won't be reached
        ]
        amount = 100
        result = lending_protocol_api.enter_lending_position(amount, "AaveV3", os.getenv("ACCOUNT_ADDRESS"), os.getenv("PRIVATE_KEY"))
        self.assertIn("USDC Approval failed:", result)
        self.mock_web3.eth.contract().functions.deposit.assert_not_called()

    def test_adjust_lending_position_withdraw_success(self):
        """Tests adjust_lending_position for successful withdrawal."""
        self.mock_web3.eth.wait_for_transaction_receipt.return_value = {'status': 1, 'transactionHash': b'\x03' * 32}
        amount = 50
        result = lending_protocol_api.adjust_lending_position(amount, "AaveV3", os.getenv("ACCOUNT_ADDRESS"), os.getenv("PRIVATE_KEY"), action="withdraw")
        self.assertIn("Successfully adjusted lending position (withdraw).", result)
        self.mock_web3.eth.contract().functions.withdraw.assert_called_once()

    def test_adjust_lending_position_deposit_success(self):
        """Tests adjust_lending_position for successful deposit."""
        self.mock_web3.eth.wait_for_transaction_receipt.side_effect = [
            {'status': 1, 'transactionHash': b'\x04' * 32}, # Approval receipt
            {'status': 1, 'transactionHash': b'\x05' * 32}  # Deposit receipt
        ]
        amount = 25
        result = lending_protocol_api.adjust_lending_position(amount, "AaveV3", os.getenv("ACCOUNT_ADDRESS"), os.getenv("PRIVATE_KEY"), action="deposit")
        self.assertIn("Successfully adjusted lending position (deposit).", result)
        self.mock_web3.eth.contract().functions.approve.assert_called_once() # Approval should be called first for deposit
        self.mock_web3.eth.contract().functions.deposit.assert_called_once()

    def test_get_current_position(self):
        """Tests the get_current_position tool."""
        account_address = os.getenv("ACCOUNT_ADDRESS")
        result = lending_protocol_api.get_current_position(account_address)
        self.assertIn("Current USDC balance in lending protocol: 100.000000 USDC", result)
        self.mock_web3.eth.contract().functions.balanceOf.assert_called_once_with(account_address, lending_protocol_api.USDC_CONTRACT_ADDRESS)

    @patch('tools.risk_assessment_api.get_volatility_index')
    @patch('random.uniform')
    def test_assess_risk_high(self, mock_random_uniform, mock_get_volatility_index):
        """Tests assess_risk for high risk scenario."""
        mock_get_volatility_index.return_value = 2.0
        mock_random_uniform.return_value = 9.0
        result = risk_assessment_api.assess_risk()
        self.assertIn("Risk Assessment: Level=High", result)

    @patch('tools.risk_assessment_api.get_volatility_index')
    @patch('random.uniform')
    def test_assess_risk_medium(self, mock_random_uniform, mock_get_volatility_index):
        """Tests assess_risk for medium risk scenario."""
        mock_get_volatility_index.return_value = 1.0
        mock_random_uniform.return_value = 6.0
        result = risk_assessment_api.assess_risk()
        self.assertIn("Risk Assessment: Level=Medium", result)

    @patch('tools.risk_assessment_api.get_volatility_index')
    @patch('random.uniform')
    def test_assess_risk_low(self, mock_random_uniform, mock_get_volatility_index):
        """Tests assess_risk for low risk scenario."""
        mock_get_volatility_index.return_value = 0.5
        mock_random_uniform.return_value = 3.0
        result = risk_assessment_api.assess_risk()
        self.assertIn("Risk Assessment: Level=Low", result)

    @patch('tools.lending_protocol_api.adjust_lending_position')
    def test_mitigate_risk(self, mock_adjust_lending_position):
        """Tests the mitigate_risk tool."""
        mock_adjust_lending_position.return_value = "Simulated withdrawal successful"
        reason = "High volatility detected"
        result = risk_assessment_api.mitigate_risk(reason)
        # Since the actual call to adjust_lending_position is commented out in mitigate_risk,
        # we check for the simulated output message.
        self.assertIn("Risk mitigation initiated for: High volatility detected.", result)
        self.assertIn("(Simulated action: Position reduction recommended.)", result)

if __name__ == '__main__':
    unittest.main()
