# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


# tests/test_agents.py
import unittest
from unittest.mock import patch, MagicMock
from agents.orchestrator_agent import orchestrator_agent
from agents.lending_agent import lending_agent
from agents.yield_optimizer_agent import yield_optimizer_agent
from agents.risk_manager_agent import risk_manager_agent
from google_adk.testing import MockLlmResponse
import asyncio

# Mock the run_async method for all agents to avoid actual LLM calls during testing
# This is a simplified approach. For more complex interactions, you might need to
# mock specific tool calls within the agents.

def mock_agent_run_async(self, instruction):
    # Return a mock response that simulates a successful LLM call
    return MockLlmResponse(content=f"Mock response for instruction: {instruction}")

class TestAgents(unittest.TestCase):

    @patch('google_adk.agents.LlmAgent.run_async', side_effect=mock_agent_run_async)
    def test_orchestrator_agent_run(self, mock_run_async):
        """Tests the orchestrator agent's run_async method."""
        instruction = "Start the lending process."
        result = orchestrator_agent.run_async(instruction)
        self.assertIsInstance(result, MockLlmResponse)
        self.assertIn("Mock response for instruction: Start the lending process.", result.content)
        mock_run_async.assert_called_once_with(instruction)

    @patch('google_adk.agents.LlmAgent.run_async', side_effect=mock_agent_run_async)
    def test_lending_agent_run(self, mock_run_async):
        """Tests the lending agent's run_async method."""
        instruction = "Enter initial position."
        result = lending_agent.run_async(instruction)
        self.assertIsInstance(result, MockLlmResponse)
        self.assertIn("Mock response for instruction: Enter initial position.", result.content)
        mock_run_async.assert_called_once_with(instruction)

    @patch('google_adk.agents.LlmAgent.run_async', side_effect=mock_agent_run_async)
    def test_yield_optimizer_agent_run(self, mock_run_async):
        """Tests the yield optimizer agent's run_async method."""
        instruction = "Optimize yield."
        result = yield_optimizer_agent.run_async(instruction)
        self.assertIsInstance(result, MockLlmResponse)
        self.assertIn("Mock response for instruction: Optimize yield.", result.content)
        mock_run_async.assert_called_once_with(instruction)

    @patch('google_adk.agents.LlmAgent.run_async', side_effect=mock_agent_run_async)
    def test_risk_manager_agent_run(self, mock_run_async):
        """Tests the risk manager agent's run_async method."""
        instruction = "Assess and mitigate risk."
        result = risk_manager_agent.run_async(instruction)
        self.assertIsInstance(result, MockLlmResponse)
        self.assertIn("Mock response for instruction: Assess and mitigate risk.", result.content)
        mock_run_async.assert_called_once_with(instruction)

    # Add more specific tests for tool interactions if needed
    # Example: Mocking a specific tool call within an agent
    @patch('tools.lending_protocol_api.enter_lending_position')
    def test_lending_agent_uses_enter_position_tool(self, mock_enter_position):
        """Tests if lending agent calls the enter_lending_position tool."""
        mock_enter_position.return_value = "Mock enter position success"
        # We need to simulate the agent deciding to call this tool.
        # This is complex as it depends on the LLM's reasoning.
        # For simplicity, we'll directly call the tool function via the agent's tool list.
        # In a real scenario, you'd test the agent's LLM output and tool selection.
        
        # This test is more of a sanity check that the tool is accessible.
        # A full test would involve mocking the LLM response that selects the tool.
        pass # Skipping direct tool call test as it bypasses LLM reasoning

if __name__ == '__main__':
    unittest.main()
